
************************************************
************************************************
*
* [Comparative Population Studies]
*								
* Nathan, M. & Pardo, I. (2019), 
* "Fertility postponement and regional patterns of dispersion in age at first birth: 
* Descriptive findings and interpretations" 
*
************************************************
************************************************


***************************
** Human Fertility Database 
** (OCT 2016) 
***************************

cd "Set directory here"

//	Import and merge datasets: 
	//	births by age of the mother and birth order 
	//	exposures of women by age
	//	ASFR by birth order
	//	TFR by birth order
	//  MAB by birth order
	//	sdMAB by birth order

import delimited "HFD\birthsRRbo.txt", delimiter(whitespace, collapse) bindquote(nobind) varnames(3) case(lower) rowrange(3) clear 
replace age="12" if age=="12-"
replace age="55" if age=="55+"
destring age, replace
sort code year age
save "births_bo.dta", replace

import delimited "HFD\exposRR.txt", delimiter(whitespace, collapse) bindquote(nobind) varnames(3) case(lower) rowrange(3) clear 
sort code year age
save "exposures.dta", replace

import delimited "HFD\asfrRRbo.txt", delimiter(whitespace, collapse) bindquote(nobind) varnames(3) case(lower) rowrange(3) clear 
replace age="12" if age=="12-"
replace age="55" if age=="55+"
destring age, replace
sort code year age
save "ASFR_bo.dta", replace

import delimited "HFD\tfrRRbo.txt", delimiter(whitespace, collapse) bindquote(nobind) varnames(3) case(lower) rowrange(3) clear 
sort code year
save "TFR_bo.dta", replace

import delimited "HFD\mabRRbo.txt", delimiter(whitespace, collapse) bindquote(nobind) varnames(3) case(lower) rowrange(3) clear 
sort code year
save "MAB_bo.dta", replace

import delimited "HFD\sdmabRRbo.txt", delimiter(whitespace, collapse) bindquote(nobind) varnames(3) case(lower) rowrange(3) clear 
sort code year
save "sdMAB_bo.dta", replace

use "births_bo.dta", clear
merge 1:1 code year age using "exposures.dta"
keep if _m==3
drop _m
merge 1:1 code year age using "ASFR_bo.dta"
drop _m
merge m:1 code year 	using "TFR_bo.dta"
drop _m
merge m:1 code year 	using "MAB_bo.dta"
drop _m
merge m:1 code year 	using "sdMAB_bo.dta"
drop _m
sort code year age

drop if year<1970		// Delete data before 1970
 

//	Identification and classification:

	//	one row per country-year

gen		onerow=1	if age==12
replace	onerow=0	if age!=12

	//	first year and final year available in data per country

by code, sort : egen float minyear = min(year)	
by code, sort : egen float maxyear = max(year)	

	//	time-span available in data per country

gen		timespan=maxyear-minyear

	
	///	list first & final year of data per country

set more off
	
list	code year	if onerow==1 & (year==minyear | year==maxyear), noobs clean

/*     
		AUT	1984	 
		AUT	2014	 
		BGR	1970	 
		BGR	2009	 
		BLR	1970	 
		BLR	2014	 
		CAN	1970	 
		CAN	2011	 
		CHE	1998	 
		CHE	2014	 
		CHL	1992	 
		CHL	2005	 
		CZE	1970	 
		CZE	2014	 
		DEUTE	1970	 
		DEUTE	2013	 
		DEUTNP	2009	 
		DEUTNP	2013	 
		DEUTW	2009	 
		DEUTW	2013	 
		ESP	1975	 
		ESP	2014	 
		EST	1970	 
		EST	2014	 
		FIN	1982	 
		FIN	2012	 
		GBRTENW	2013	 
		GBR_NIR	1997	 
		GBR_NIR	2013	 
		GBR_NP	2013	 
		GBR_SCO	2013	 
		HUN	1970	 
		HUN	2014	 
		ISL	1990	 
		ISL	2012	 
		ITA	2004	 
		ITA	2012	 
		JPN	1970	 
		JPN	2012	 
		LTU	1970	 
		LTU	2014	 
		NLD	1970	 
		NLD	2012	 
		NOR	1970	 
		NOR	2014	 
		PRT	1970	 
		PRT	2012	 
		RUS	1970	 
		RUS	2014	 
		SVK	1970	 
		SVK	2009	 
		SVN	1983	 
		SVN	2014	 
		SWE	1970	 
		SWE	2014	 
		TWN	1976	 
		TWN	2014	 
		UKR	1970	 
		UKR	2013	 
		USA	1970	 
		USA	2014	 */



	//	Selected countries are grouped into 8 regions by region
		
*************************************************************
*	Western Europe		Northern Europe		Southern Europe
*	AUT					FIN					ESP
*	NLD					NOR					PRT
*						SWE					
*								
*	Post-soviet count.	Central & Eastern Europe				
*	BLR					BGR
*	RUS					CZE
*	UKR					HUN
*												
*	North America		South America		East Asia
*	CAN					CHL					JPN
*	USA										TWN
*************************************************************	

gen		region=.
replace	region=1	if code=="AUT" | code=="NLD"		
replace	region=2	if code=="FIN" | code=="NOR" | code=="SWE"
replace	region=3	if code=="ESP" | code=="PRT" 
replace	region=4	if code=="BLR" | code=="RUS" | code=="UKR" 
replace	region=5	if code=="BGR" | code=="CZE" | code=="HUN"
replace	region=6	if code=="CAN" | code=="USA"
replace	region=7	if code=="CHL"
replace	region=8	if code=="JPN" | code=="TWN" 

label 	define region 	1 "Western Europe" 	/// 
						2 "Northern Europe" /// 
						3 "Southern Europe" /// 
						4 "Post-soviet countries" /// 
						5 "Central & Eastern Europe" /// 
						6 "North America" 	/// 
						7 "South America" 	/// 
						8 "East Asia"
label	values region region  
 

keep if region!=.		// Keep selected countries 
 
 
//	Distribution and dispersion variables 


	// (%) share of first births by age

by code year, sort: gen rel_asfr1			= asfr1/tfr1*100	

by code year, sort: egen checkrel			= total(rel_asfr1)	// checking if sum(asfr1) = 1	
sum checkrel, detail
drop checkrel

label var rel_asfr1 "%ASFR1"


	// (%) share of first births by age group
	
by code year, sort: egen bef20asfr1 		= total (asfr1)		if age<20 
by code year, sort: egen bet20_29asfr1 		= total (asfr1)		if age>=20 & age<=29
by code year, sort: egen aft29asfr1 		= total (asfr1)		if age>=30

by code year, sort: egen asfr1_20 		= max (bef20asfr1)		
by code year, sort: egen asfr1_20_29 	= max (bet20_29asfr1)
by code year, sort: egen asfr1_30 		= max (aft29asfr1)	

gen rel_20		=	asfr1_20/(asfr1_20+asfr1_20_29+asfr1_30)*100
gen rel_30		=	asfr1_30/(asfr1_20+asfr1_20_29+asfr1_30)*100

label var rel_20 "% FBR before age 20"
label var rel_30 "% FBR after age 29"


	// Ratio <20/>29

gen ratio_2030	=	rel_20/rel_30

label var ratio_2030 "Ratio FBR before 20 & after 30"


	// Coefficient of variation

gen cv1			=	sdmab1/mab1

label var cv1 "Coefficient of variation of age at first birth"


********************************************
*										
*	Save two files: 	
*										

sort code year
save "HFD_1970-2015", replace			// 1. Variables by country, year and age of the mother

keep if onerow==1
keep code year minyear maxyear timespan region tfr tfr1 mab1 sdmab1 rel_20 rel_30 ratio_2030 cv1
save "HFD_fbirths_1970-2015", replace	// 2. Summary measures by country and year

********************************************************************************


****************************
* Human fertility Collection
* (OCT 2016)
****************************


cd "Set directory here"

//	Import dataset:

	//	ASFR by birth order
		// Keep selected countries:	
		/*	- UK: England & Wales
			- Greece
			- Uruguay
		*/

		
import delimited "HFC\HFC_ASFR_BO.txt", delimiter(comma) clear

keep if country=="GBRTENW" | country=="    GRC" | country=="    URY"
replace country="GRC"	if country=="    GRC"
replace country="URY"	if country=="    URY"

destring asfr, replace
destring asfr3, replace
destring asfr3p, replace
destring asfr4, replace
destring asfr4p, replace
destring asfr5p, replace

rename country code
rename year1 year

keep if agedef=="   ACY"						

drop if year<1970		// Drop data before 1970
					
drop region urban origin year2 agedef vitality collection sourcetype refcode note	// Drop surplus variables

drop asfr asfr2-asfr5p	// Keep fertility rates for first births


//	Basic computations:

	//	one row per country-year variable

by code year, sort : egen float minage = min(age)
by code year, sort : egen float maxage = max(age)

gen		onerow=1	if age==minage
replace	onerow=0	if age!=minage

	//	first year and final year available in data per country

by code, sort : egen float minyear = min(year)	
by code, sort : egen float maxyear = max(year)	

	//	time-span available in data per country

gen		timespan=maxyear-minyear

	///	list 

list	code year minage maxage 	if onerow==1 & (year==minyear | year==maxyear), noobs clean

/*	   
       code   year   minage   maxage  
    GBRTENW   1970       15       45  
    GBRTENW   2013       14       50  
        GRC   1970       15       49  
        GRC   2008       15       49  
        URY   1978       10       45  // URY 5-age-int 1978-1993
        URY   2011       12       50  // URY 1-age-int 1996-2011
*/

	//	Compute regions
		
		// Same criteria as with the HFD countries
		
************************************************************************
*	Western Europe		Northern Europe		Southern Europe
*	AUT					FIN					ESP
*	GBRTENW				NOR					GRC
*	NLD					SWE					PRT
*	
*	Post-soviet count.	Central & Eastern Europe				
*	BLR					BGR
*	RUS					CZE
*	UKR					HUN
*											
*	North America		South America		East Asia
*	CAN					CHL					JPN
*	USA					URY					TWN
************************************************************************	

gen		region=.
replace	region=1	if code=="GBRTENW"
replace	region=3	if code=="GRC"
replace	region=7	if code=="URY"									

label 	define region 	1 "Western Europe" 	/// 
						2 "Northern Europe" /// 
						3 "Southern Europe" /// 
						4 "Post-soviet countries" /// 
						5 "Central & Eastern Europe" /// 
						6 "North America" 	/// 
						7 "South America" 	/// 
						8 "East Asia"

label	values region region  


//	Summary measures
	
	// Open age intervals?

		/*	Possible options:
			1. Compute same age interval for every observation of the age structure
			2. Compute 1 year when age interval is 5-year age group (e.g. when minor age is 14)
			3. Compute specific values (e.g. 2.5 years at age 45)
		*/

replace ageint = 1		if code=="GBRTENW" 	///
						| code=="GRC"		///
						| code=="URY" 		///
						& (ageint==-99 | ageint==99)
						
	//	TFR1		
							
gen		asfr1_wg=asfr1*ageint	

by code year, sort : egen float tfr1 = total(asfr1_wg)	


	//	MAB1		

gen		agefx	=	(age+0.5)*asfr1	if ageint==1 
replace	agefx	=	(age+2.5)*asfr1 if ageint==5 

by code year, sort : egen mab1 = total(agefx)
 
gen		mab1a	=	mab1/tfr1		  		
gen		mab1b	=	mab1/(tfr1/5)		
replace mab1	=	mab1a
replace	mab1	=	mab1b	if (code=="URY" & (year>=1978 & year<=1993))


	// sdMAB1
	
gen		sdMAB_a	=	((age+0.5)^2)*(asfr1/tfr1)		if ageint==1
replace	sdMAB_a	=	((age+2.5)^2)*(asfr1/tfr1*5)	if ageint==5

gen		sdMAB_b	=	(age+0.5)*(asfr1/tfr1)			if ageint==1
replace	sdMAB_b	=	(age+2.5)*(asfr1/tfr1*5)		if ageint==5

by code year, sort :	egen sdMAB_A	=	total(sdMAB_a)
by code year, sort :	egen sdMAB_B	=	total(sdMAB_b)
replace	sdMAB_B	=	sdMAB_B^2
gen		sdMAB_C	=	sdMAB_A - sdMAB_B
gen 	sdmab1	=	sqrt(sdMAB_C)	


//	(complementary) Dispersion measures


	// % first births by single age

by code year, sort: gen rel_asfr1			= asfr1/tfr1*100	

by code year, sort: egen checkrel			= total(rel_asfr1)	// checking if sum(asfr1)=1	
sum checkrel, detail
drop checkrel

label var rel_asfr1 "%ASFR1"


	// % first births by age-group

by code year, sort: egen bef20asfr1 		= total (asfr1)		if age<20 
by code year, sort: egen bet20_29asfr1 		= total (asfr1)		if age>=20 & age<=29
by code year, sort: egen aft29asfr1			= total (asfr1)		if age>=30

by code year, sort: egen asfr1_20 		= max (bef20asfr1)		
by code year, sort: egen asfr1_20_29 	= max (bet20_29asfr1)
by code year, sort: egen asfr1_30 		= max (aft29asfr1)				

drop bef20asfr1-aft29asfr1

gen rel_20		=	asfr1_20/(asfr1_20+asfr1_20_29+asfr1_30)*100
gen rel_30		=	asfr1_30/(asfr1_20+asfr1_20_29+asfr1_30)*100

label var rel_20 "% FBR before age 20"
label var rel_30 "% FBR after age 29"


	// ratio <20/>29

gen ratio_2030	=	rel_20/rel_30

label var ratio_2030 "Ratio FBR before 20 & after 30"


	// coefficient of variation

gen cv1			=	sdmab1/mab1

label var cv1 "Coefficient of variation of age at first birth"


********************************************************************************
*										
*	Save two files: 	
*										


sort 	code year
save "HFC_1970-2015", replace			// 1. Variables by country, year and age of the mother

keep 	if onerow==1
keep	code year minyear maxyear timespan region tfr tfr1 mab1 sdmab1 rel_20 rel_30 ratio_2030 cv1
save "HFC_fbirths_1970-2015", replace	// 2. Summary measures by country and year

********************************************************************************


***********************************************
** COMBINE DATASETS & GENERATE NEW VARIABLES **
***********************************************

	
	// Append HFD & HFC - Datasets by age of the mother -
	
cap cd "set directory here"
set more off

use "HFD_1970-2015", clear

append using "HFC_1970-2015.dta"

keep code year age asfr1 tfr tfr1 mab1 sdmab1 onerow minyear maxyear timespan region rel_asfr1 asfr1_20 asfr1_20_29 asfr1_30 rel_20 rel_30 ratio_2030 cv1

gen country=code
replace country="Finland" if code=="FIN"	
replace country="Norway" if code=="NOR"	
replace country="Sweden" if code=="SWE"	
replace country="Austria" if code=="AUT"	
replace country="England & Wales" if code=="GBRTENW"	
replace country="The Netherlands" if code=="NLD"	
replace country="Spain" if code=="ESP"	
replace country="Greece" if code=="GRC"	
replace country="Portugal" if code=="PRT"	
replace country="Belarus" if code=="BLR"	
replace country="Russia" if code=="RUS"	
replace country="Ukraine" if code=="UKR"
replace country="Bulgaria" if code=="BGR"	
replace country="Czech Republic" if code=="CZE"	
replace country="Hungary" if code=="HUN"
replace country="Canada" if code=="CAN"	
replace country="United States" if code=="USA"	
replace country="Chile" if code=="CHL"	
replace country="Uruguay" if code=="URY"	
replace country="Japan" if code=="JPN"
replace country="Taiwan" if code=="TWN"	
	
save "HFD-HFC_1970-2015", replace


	//	Complementary measures of dispersion (percentiles, interquartile range, skewness, kurtosis)
		
			/*	Remember: STATA results window 
				shoud be maximized first before running the code */  
			
foreach C in "AUT" "BGR" "BLR" "CAN" "CHL" "CZE" "ESP" "FIN" "GBRTENW" "GRC" "HUN" "JPN" "NLD" "NOR" "PRT" "RUS" "SWE" "TWN" "UKR" "URY" "USA" {	 
	logout, save(q_`C')  replace: tabstat age [aw=asfr1] if code=="`C'", by(year) s(p10 p25 p50 p75 p90 iqr skewness kurtosis) noseparator
}
*
		
foreach C in "AUT" "BGR" "BLR" "CAN" "CHL" "CZE" "ESP" "FIN" "GBRTENW" "GRC" "HUN" "JPN" "NLD" "NOR" "PRT" "RUS" "SWE" "TWN" "UKR" "URY" "USA" {	 
	import delimited "q_`C'.txt", delimiter(space, collapse) varnames(4) rowrange(7) colrange(2) clear
	drop v2
	destring year, force replace
	drop if year==.
	gen code="`C'"
	save "`C'.dta", replace
}
*

use "AUT.dta", clear
foreach C in "BGR" "BLR" "CAN" "CHL" "CZE" "ESP" "FIN" "GBRTENW" "GRC" "HUN" "JPN" "NLD" "NOR" "PRT" "RUS" "SWE" "TWN" "UKR" "URY" "USA" {	 
	append using "`C'.dta"
}
*

sort code year

save "HFD-HFC_fbirths_1970-2015_disp", replace


	// Append HFD & HFC - Datasets with aggregated data - 
	
use "HFD_fbirths_1970-2015", clear

append using "HFC_fbirths_1970-2015.dta"

gen country=code
replace country="Finland" if code=="FIN"	
replace country="Norway" if code=="NOR"	
replace country="Sweden" if code=="SWE"	
replace country="Austria" if code=="AUT"	
replace country="England & Wales" if code=="GBRTENW"	
replace country="The Netherlands" if code=="NLD"	
replace country="Spain" if code=="ESP"	
replace country="Greece" if code=="GRC"	
replace country="Portugal" if code=="PRT"	
replace country="Belarus" if code=="BLR"	
replace country="Russia" if code=="RUS"	
replace country="Ukraine" if code=="UKR"
replace country="Bulgaria" if code=="BGR"	
replace country="Czech Republic" if code=="CZE"	
replace country="Hungary" if code=="HUN"
replace country="Canada" if code=="CAN"	
replace country="United States" if code=="USA"	
replace country="Chile" if code=="CHL"	
replace country="Uruguay" if code=="URY"	
replace country="Japan" if code=="JPN"
replace country="Taiwan" if code=="TWN"	

		//	NEW VARIABLES: Pace of MAB1 increase and onset of PT

sort code year

by code: gen pace = mab1[_n+1] - mab1 	if year!=minyear

by code: gen cumm = pace + pace[_n-1] + pace[_n-2]	

by code: gen cumm_3 = 1		if cumm>=0.3 & cumm!=.
by code: gen cumm_4 = 1		if cumm>=0.4 & cumm!=.  
by code: gen cumm_5 = 1		if cumm>=0.5 & cumm!=.

by code: gen threeyear = 1	if	(pace>0 & pace!=.) ///
							& 	(pace[_n-1]>0 & pace[_n-1]!=.) ///
							& 	(pace[_n-2]>0 & pace[_n-2]!=.) 

							
by code: egen KBO_criteria = min(year)	if cumm_3==1 & threeyear==1							
by code: replace KBO_criteria = KBO_criteria - 2  	// Matches with Kohler, Billari and Ortega 2002
by code: egen KBO_year = max(KBO_criteria)
		 gen  KBO_val  = mab1			if year==KBO_year 
by code: egen KBO_mab1 = max(KBO_val) 
		 gen  KBO_sd  = sdmab1			if year==KBO_year 
by code: egen KBO_sdmab1 = max(KBO_sd) 

				
by code: egen Sob_criteria = min(year)	if cumm_5==1 & threeyear==1
by code: replace Sob_criteria = Sob_criteria - 2    // Do not fully match with Sobotka 2004
by code: egen Sob_year = max(Sob_criteria)
		 gen  Sob_val  = mab1			if year==Sob_year 
by code: egen Sob_mab1 = max(Sob_val) 
		 gen  Sob_sd  = sdmab1			if year==Sob_year 
by code: egen Sob_sdmab1 = max(Sob_sd) 
				
				
by code: egen NP_criteria =  min(year)	if cumm_4==1 & threeyear==1	
by code: replace NP_criteria = NP_criteria - 2		// Nathan-Pardo proposal (nor 3 years neither 5 years of continuos increase, but 4 years)
by code: egen NP_year = max(NP_criteria)
		 gen  NP_val  = mab1			if year==NP_year 
by code: egen NP_mab1 = max(NP_val) 
		 gen  NP_sd  = sdmab1			if year==NP_year 
by code: egen NP_sdmab1 = max(NP_sd) 

drop cumm-threeyear
				
sort region code year
list region code  KBO_year NP_year Sob_year  KBO_mab1 NP_mab1 Sob_mab1  KBO_sdmab1 NP_sdmab1 Sob_sdmab1 if minyear==year, noobs


		//	Merge dataset with other dispersion measures
		
merge 1:1 code year using "HFD-HFC_fbirths_1970-2015_disp"
drop _m
		
		//	SAVE FILE

save "HFD-HFC_fbirths_1970-2015", replace

********************************************************************************

*****************************
** EXPORT DATASETs TO EXCEL **
*****************************
	
cap cd "Set directory here"

use		"HFD-HFC_1970-2015", clear
keep code age year mab1 sdmab1 region rel_20 rel_30 cv1 country 
export excel using "CPoS_HFD-HFC_1970-2015.xlsx", sheet("Data") firstrow(variables) replace	


use		"HFD-HFC_fbirths_1970-2015", clear
keep code year mab1 sdmab1 region rel_20 rel_30 cv1 country   
export excel using "CPoS_HFD-HFC_fbirths_1970-2015.xlsx", sheet("Data") firstrow(variables) replace	

********************************************************************************
	

******************
** STATA CHARTS **
******************

cap cd "Set directory here"

use		"HFD-HFC_fbirths_1970-2015", clear

	
		// Scatter: MAB1 & sdMAB1 by region and country

twoway	(scatter sdmab1 mab1 if code=="FIN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter sdmab1 mab1 if code=="NOR", msymbol(T) connect(l) color(emidblue)) ///
		(scatter sdmab1 mab1 if code=="SWE", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(3(1)7, grid) xlabel(20(2)32, grid) title(Northern Europe) ///
		legend(label(1 "Finland") label(2 "Norway") label(3 "Sweden") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-sdMAB1_NE_thecon_symb.emf", replace

twoway	(scatter sdmab1 mab1 if code=="AUT", msymbol(O) connect(l) color(eltblue)) ///
		(scatter sdmab1 mab1 if code=="GBRTENW", msymbol(T) connect(l) color(emidblue)) ///
		(scatter sdmab1 mab1 if code=="NLD", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(3(1)7, grid) xlabel(20(2)32, grid) title(Western Europe) ///
		legend(label(1 "Austria") label(2 "England&Wales") label(3 "The Netherlands") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-sdMAB1_WE_thecon_symb.emf", replace

twoway	(scatter sdmab1 mab1 if code=="ESP", msymbol(O) connect(l) color(eltblue)) ///
		(scatter sdmab1 mab1 if code=="GRC", msymbol(T) connect(l) color(emidblue)) ///
		(scatter sdmab1 mab1 if code=="PRT", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(3(1)7, grid) xlabel(20(2)32, grid) title(Southern Europe) ///
		legend(label(1 "Spain") label(2 "Greece") label(3 "Portugal") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-sdMAB1_SE_thecon_symb.emf", replace
		
twoway	(scatter sdmab1 mab1 if code=="BGR", msymbol(O) connect(l) color(eltblue)) ///
		(scatter sdmab1 mab1 if code=="CZE", msymbol(T) connect(l) color(emidblue)) ///
		(scatter sdmab1 mab1 if code=="HUN", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(3(1)7, grid) xlabel(20(2)32, grid) title(Central & Eastern Europe) ///
		legend(label(1 "Bulgaria") label(2 "Czech Rep.") label(3 "Hungary") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-sdMAB1_CEE_thecon_symb.emf", replace

twoway	(scatter sdmab1 mab1 if code=="BLR", msymbol(O) connect(l) color(eltblue)) ///
		(scatter sdmab1 mab1 if code=="RUS", msymbol(T) connect(l) color(emidblue)) ///
		(scatter sdmab1 mab1 if code=="UKR", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(3(1)7, grid) xlabel(20(2)32, grid) title(Post-Soviet countries) ///
		legend(label(1 "Belarus") label(2 "Russia") label(3 "Ukraine") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-sdMAB1_PSC_thecon_symb.emf", replace

twoway	(scatter sdmab1 mab1 if code=="JPN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter sdmab1 mab1 if code=="TWN", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(3(1)7, grid) xlabel(20(2)32, grid) title(East Asia) ///
		legend(label(1 "Japan") label(2 "Taiwan") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-sdMAB1_EA_thecon_symb.emf", replace
		
twoway	(scatter sdmab1 mab1 if code=="CAN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter sdmab1 mab1 if code=="USA", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(3(1)7, grid) xlabel(20(2)32, grid) title(North America) ///
		legend(label(1 "Canada") label(2 "United States") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-sdMAB1_NA_thecon_symb.emf", replace

twoway	(scatter sdmab1 mab1 if code=="CHL", msymbol(O) connect(l) color(eltblue)) ///
		(scatter sdmab1 mab1 if code=="URY", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(3(1)7, grid) xlabel(20(2)32, grid) title(South America) ///
		legend(label(1 "Chile") label(2 "Uruguay") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-sdMAB1_SA_thecon_symb.emf", replace
		
		
		// Scatter: % first birth rates -20 & 30+ by region and country		
		
twoway	(scatter rel_20 rel_30 if code=="FIN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter rel_20 rel_30 if code=="NOR", msymbol(T) connect(l) color(emidblue)) ///
		(scatter rel_20 rel_30 if code=="SWE", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(10)40, grid) xlabel(0(10)60, grid) title(Northern Europe) ///
		legend(label(1 "Finland") label(2 "Norway") label(3 "Sweden") rows(1)) scheme(s1color)
		graph export "Charts/%ASFR1_20_30_NE_thecon_symb.emf", replace

twoway	(scatter rel_20 rel_30 if code=="AUT", msymbol(O) connect(l) color(eltblue)) ///
		(scatter rel_20 rel_30 if code=="GBRTENW", msymbol(T) connect(l) color(emidblue)) ///
		(scatter rel_20 rel_30 if code=="NLD", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(10)40, grid) xlabel(0(10)60, grid) title(Western Europe) ///
		legend(label(1 "Austria") label(2 "England&Wales") label(3 "The Netherlands") rows(1)) scheme(s1color)
		graph export "Charts/%ASFR1_20_30_WE_thecon_symb.emf", replace

twoway	(scatter rel_20 rel_30 if code=="ESP", msymbol(O) connect(l) color(eltblue)) ///
		(scatter rel_20 rel_30 if code=="GRC", msymbol(T) connect(l) color(emidblue)) ///
		(scatter rel_20 rel_30 if code=="PRT", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(10)40, grid) xlabel(0(10)60, grid) title(Southern Europe) ///
		legend(label(1 "Spain") label(2 "Greece") label(3 "Portugal") rows(1)) scheme(s1color)
		graph export "Charts/%ASFR1_20_30_SE_thecon_symb.emf", replace

twoway	(scatter rel_20 rel_30 if code=="BGR", msymbol(O) connect(l) color(eltblue)) ///
		(scatter rel_20 rel_30 if code=="CZE", msymbol(T) connect(l) color(emidblue)) ///
		(scatter rel_20 rel_30 if code=="HUN", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(10)40, grid) xlabel(0(10)60, grid) title(Central & Eastern Europe) ///
		legend(label(1 "Bulgaria") label(2 "Czech Rep.") label(3 "Hungary") rows(1)) scheme(s1color)
		graph export "Charts/%ASFR1_20_30_CEE_thecon_symb.emf", replace

twoway	(scatter rel_20 rel_30 if code=="BLR", msymbol(O) connect(l) color(eltblue)) ///
		(scatter rel_20 rel_30 if code=="RUS", msymbol(T) connect(l) color(emidblue)) ///
		(scatter rel_20 rel_30 if code=="UKR", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(10)40, grid) xlabel(0(10)60, grid) title(Post-Soviet countries) ///
		legend(label(1 "Belarus") label(2 "Russia") label(3 "Ukraine") rows(1)) scheme(s1color)
		graph export "Charts/%ASFR1_20_30_PSC_thecon_symb.emf", replace
		
twoway	(scatter rel_20 rel_30 if code=="JPN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter rel_20 rel_30 if code=="TWN", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(0(10)40, grid) xlabel(0(10)60, grid) title(East Asia) ///
		legend(label(1 "Japan") label(2 "Taiwan") rows(1)) scheme(s1color)
		graph export "Charts/%ASFR1_20_30_EA_thecon_symb.emf", replace

twoway	(scatter rel_20 rel_30 if code=="CAN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter rel_20 rel_30 if code=="USA", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(0(10)40, grid) xlabel(0(10)60, grid) title(North America) ///
		legend(label(1 "Canada") label(2 "United States") rows(1)) scheme(s1color)
		graph export "Charts/%ASFR1_20_30_NA_thecon_symb.emf", replace

twoway	(scatter rel_20 rel_30 if code=="CHL", msymbol(O) connect(l) color(eltblue)) ///
		(scatter rel_20 rel_30 if code=="URY", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(0(10)40, grid) xlabel(0(10)60, grid) title(South America) ///
		legend(label(1 "Chile") label(2 "Uruguay") rows(1)) scheme(s1color)
		graph export "Charts/%ASFR1_20_30_SA_thecon_symb.emf", replace		


		// Scatter: MAB1 and cv1 by region and country		
		
twoway	(scatter cv1 mab1 if code=="FIN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter cv1 mab1 if code=="NOR", msymbol(T) connect(l) color(emidblue)) ///
		(scatter cv1 mab1 if code=="SWE", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(.1).3, grid) xlabel(20(2)32, grid) title(Northern Europe) ///
		legend(label(1 "Finland") label(2 "Norway") label(3 "Sweden") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-CV1_NE_thecon_symb.emf", replace

twoway	(scatter cv1 mab1 if code=="AUT", msymbol(O) connect(l) color(eltblue)) ///
		(scatter cv1 mab1 if code=="GBRTENW", msymbol(T) connect(l) color(emidblue)) ///
		(scatter cv1 mab1 if code=="NLD", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(.1).3, grid) xlabel(20(2)32, grid) title(Western Europe) ///
		legend(label(1 "Austria") label(2 "England&Wales") label(3 "The Netherlands") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-CV1_WE_thecon_symb.emf", replace

twoway	(scatter cv1 mab1 if code=="ESP", msymbol(O) connect(l) color(eltblue)) ///
		(scatter cv1 mab1 if code=="GRC", msymbol(T) connect(l) color(emidblue)) ///
		(scatter cv1 mab1 if code=="PRT", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(.1).3, grid) xlabel(20(2)32, grid) title(Southern Europe) ///
		legend(label(1 "Spain") label(2 "Greece") label(3 "Portugal") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-CV1_SE_thecon_symb.emf", replace
		
twoway	(scatter cv1 mab1 if code=="BGR", msymbol(O) connect(l) color(eltblue)) ///
		(scatter cv1 mab1 if code=="CZE", msymbol(T) connect(l) color(emidblue)) ///
		(scatter cv1 mab1 if code=="HUN", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(.1).3, grid) xlabel(20(2)32, grid) title(Central & Eastern Europe) ///
		legend(label(1 "Bulgaria") label(2 "Czech Rep.") label(3 "Hungary") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-CV1_CEE_thecon_symb.emf", replace
		
twoway	(scatter cv1 mab1 if code=="BLR", msymbol(O) connect(l) color(eltblue)) ///
		(scatter cv1 mab1 if code=="RUS", msymbol(T) connect(l) color(emidblue)) ///
		(scatter cv1 mab1 if code=="UKR", msymbol(S) connect(l) color(edkblue)) /// 
		, ylabel(0(.1).3, grid) xlabel(20(2)32, grid) title(Post-Soviet countries) ///
		legend(label(1 "Belarus") label(2 "Russia") label(3 "Ukraine") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-CV1_PSC_thecon_symb.emf", replace
		
twoway	(scatter cv1 mab1 if code=="JPN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter cv1 mab1 if code=="TWN", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(0(.1).3, grid) xlabel(20(2)32, grid) title(East Asia) ///
		legend(label(1 "Japan") label(2 "Taiwan") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-CV1_EA_thecon_symb.emf", replace
		
twoway	(scatter cv1 mab1 if code=="CAN", msymbol(O) connect(l) color(eltblue)) ///
		(scatter cv1 mab1 if code=="USA", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(0(.1).3, grid) xlabel(20(2)32, grid) title(North America) ///
		legend(label(1 "Canada") label(2 "United States") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-CV1_NA_thecon_symb.emf", replace
		
twoway	(scatter cv1 mab1 if code=="CHL", msymbol(O) connect(l) color(eltblue)) ///
		(scatter cv1 mab1 if code=="URY", msymbol(T) connect(l) color(emidblue)) ///
		, ylabel(0(.1).3, grid) xlabel(20(2)32, grid) title(South America) ///
		legend(label(1 "Chile") label(2 "Uruguay") rows(1)) scheme(s1color)
		graph export "Charts/MAB1-CV1_SA_thecon_symb.emf", replace
		

// Patterns of ASFR1 shape in selected countries and years

cap cd "Set directory here"

use "HFD-HFC_1970-2015", clear


	// 1st group
	
		// Sweden

twoway	(line rel_asfr1 age if code=="SWE" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="SWE" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="SWE" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="SWE" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel (0(2)14) title("Sweden") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_SWE_cpos.emf", replace
	
		// The Netherlands

twoway	(line rel_asfr1 age if code=="NLD" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="NLD" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="NLD" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="NLD" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel (0(2)14) title("The Netherlands") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_NLD_cpos.emf", replace

			
	// 2nd group
	
		// Czech Republic

twoway	(line rel_asfr1 age if code=="CZE" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="CZE" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="CZE" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="CZE" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel(0(2)14) title("Czech Rep.") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_CZE_cpos.emf", replace

		// Hungary

twoway	(line rel_asfr1 age if code=="HUN" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="HUN" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="HUN" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="HUN" & year==2009, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel(0(2)14) title("Hungary") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2009") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_HUN_cpos.emf", replace		

		// Russia

twoway	(line rel_asfr1 age if code=="RUS" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="RUS" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="RUS" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="RUS" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel(0(2)14) title("Russia") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_RUS_cpos.emf", replace		
				
		// Japan

twoway	(line rel_asfr1 age if code=="JPN" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="JPN" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="JPN" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="JPN" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel(0(2)14) title("Japan") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010")  rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_JPN_cpos.emf", replace
		
		// Taiwan

twoway	(line rel_asfr1 age if code=="TWN" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="TWN" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="TWN" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="TWN" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel(0(2)14) title("Taiwan") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_TWN_cpos.emf", replace	
		
		
	// 3rd group
				
		// Greece

twoway	(line rel_asfr1 age if code=="GRC" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="GRC" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="GRC" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="GRC" & year==2008, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel(0(2)14) title("Greece") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2008") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_GRC_cpos.emf", replace
	
		// Portugal

twoway	(line rel_asfr1 age if code=="PRT" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="PRT" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="PRT" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="PRT" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel(0(2)14) title("Portugal") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_PRT_cpos.emf", replace

		// Spain

twoway	(line rel_asfr1 age if code=="ESP" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="ESP" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="ESP" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="ESP" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel(0(2)14) title("Spain") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_ESP_cpos.emf", replace
		
		
	// 4rd group

		// England & Wales

twoway	(line rel_asfr1 age if code=="GBRTENW" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="GBRTENW" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="GBRTENW" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="GBRTENW" & year==2007, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel (0(2)14) title("England & Wales") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2007") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_E&W_cpos.emf", replace

		// United States

twoway	(line rel_asfr1 age if code=="USA" & year==1980, lpattern(shortdash)	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="USA" & year==1990, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="USA" & year==2000, lpattern(longdash)  	connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="USA" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel (0(2)14) title("United States") ///
		legend(label(1 "1980") label(2 "1990") label(3 "2000") label(4 "2010") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_USA_cpos.emf", replace

		// Chile

twoway	(line rel_asfr1 age if code=="CHL" & year==0) ///
		(line rel_asfr1 age if code=="CHL" & year==1992, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="CHL" & year==2005, lpattern(longdash)		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="CHL" & year==0) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel (0(2)14) title("Chile") ///
		legend(order(2 3) label(2 "1992") label(3 "2005") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_CHL_cpos.emf", replace
		
		// Uruguay

twoway	(line rel_asfr1 age if code=="URY" & year==0) ///
		(line rel_asfr1 age if code=="URY" & year==1996, lpattern(dash)  		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="URY" & year==2001, lpattern(longdash)		connect (L) lwidth(thick)) ///
		(line rel_asfr1 age if code=="URY" & year==2010, lpattern(solid)  		connect (L) lwidth(thick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel (0(2)14) title("Uruguay") ///
		legend(order(2 3 4) label(2 "1996") label(3 "2005") label(4 "2011") rows(1)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_URY_cpos.emf", replace

				
	// Selected countries with fixed MAB1
	
		// MAB1==23.7

twoway	(line rel_asfr1 age if code=="NOR" & year==1977, lpattern(solid) 		lcolor(gs15)  		connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="GBRTENW" & year==1975, lpattern(dash) 	lcolor(gs12) 		connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="TWN" & year==1978, lpattern(shortdash)	lcolor(gs9)	connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="USA" & year==1982, lpattern(dash)	lcolor(gs6)	connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="CHL" & year==2003, lpattern(longdash)	lcolor(gs3)  		connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="URY" & year==1996, lpattern(solid)	lcolor(gs0) 		connect (L) lwidth(medthick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel (0(2)14) title(MAB1=23.7) ///
		legend(label (1 "Norway 1977") label (2 "Eng. & Wal. 1975") label(3 "Taiwan 1978") label(4 "U.S. 1982") label(5 "Chile 2003") label(6 "Uruguay 1996") rows(2)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_23_cpos.emf", replace

		// MAB1==24.7

twoway	(line rel_asfr1 age if code=="NOR" & year==1984, lpattern(solid) 		lcolor(gs15)  		connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="GBRTENW" & year==1986, lpattern(dash) 	lcolor(gs12) 		connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="TWN" & year==1985, lpattern(shortdash)	lcolor(gs9)	connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="USA" & year==1996, lpattern(dash)	lcolor(gs6)	connect (L) lwidth(medthick)) ///
		(line rel_asfr1 age if code=="URY" & year==2011, lpattern(solid)	lcolor(gs0) 		connect (L) lwidth(medthick)) ///
		if (age>=15 & age<=45), xlabel(15(5)45) ylabel (0(2)14) title(MAB1=24.7) ///
		legend(label (1 "Norway 1984") label (2 "Eng. & Wal. 1986") label (3 "Taiwan 1985") label(4 "U.S. 1996") label(5 "Uruguay 2011") rows(2)) scheme(s1mono) 
		graph export "Charts/GRAPH %asfr1_24_cpos.emf", replace
		

********************************************************************************		
********************************************************************************		
	

***********
***     ***
*** END ***
***     ***
*********** 

